"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.authenticate = authenticate;
exports.authorizeAdmin = authorizeAdmin;
const jwt_1 = require("../utils/jwt");
/**
 * Middleware to authenticate requests using a JWT token.
 * - Checks for the presence of a Bearer token in the Authorization header.
 * - Verifies the token and attaches userId and role to the request object.
 * - Returns 401 Unauthorized if the token is missing, invalid, or expired.
 */
function authenticate(req, res, next) {
    const authHeader = req.headers.authorization;
    if (!authHeader?.startsWith("Bearer ")) {
        return res
            .status(401)
            .json({ error: { message: "No token provided", code: "UNAUTHORIZED" } });
    }
    const token = authHeader.split(" ")[1];
    try {
        const decoded = (0, jwt_1.verifyToken)(token);
        req.userId = decoded.id;
        req.role = decoded.role;
        next();
    }
    catch {
        return res.status(401).json({
            error: { message: "Invalid or expired token", code: "UNAUTHORIZED" },
        });
    }
}
/**
 * Middleware to authorize admin-only routes.
 * - Checks if the authenticated user's role is "ADMIN".
 * - Returns 403 Forbidden if the user is not an admin.
 */
function authorizeAdmin(req, res, next) {
    if (req.role !== "ADMIN") {
        return res
            .status(403)
            .json({ error: { message: "Admin access required", code: "FORBIDDEN" } });
    }
    next();
}
