"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.usersController = void 0;
const prisma_1 = require("../lib/prisma");
const client_1 = require("@prisma/client");
const jwt_1 = require("../lib/jwt");
exports.usersController = {
    // ✅ get user info
    profile: async (req, res) => {
        try {
            const userInfo = await prisma_1.prisma.user.findUnique({
                where: { id: req.userId },
                select: {
                    id: true,
                    phone: true,
                    name: true,
                    role: true,
                    inviteCode: true,
                    invitedBy: true,
                    invitedAt: true,
                    createdAt: true,
                    profile: {
                        select: {
                            currentBalance: true,
                            referralIncome: true,
                            vip: {
                                select: {
                                    name: true,
                                    dailyIncome: true,
                                    incomePerTask: true,
                                    commission: true,
                                },
                            },
                        },
                    },
                },
            });
            if (!userInfo) {
                return res
                    .status(404)
                    .json({ error: "User not found", success: false });
            }
            return res.status(200).json({ user: userInfo, success: true });
        }
        catch (error) {
            console.error("User info error:", error);
            return res.status(500).json({ error: "Internal server error" });
        }
    },
    // ✅ update user name
    updateName: async (req, res) => {
        try {
            const { token } = req.body;
            const user = await (0, jwt_1.verifyToken)(token);
            if (!user) {
                return new Response(JSON.stringify({ error: "Unauthorized" }), {
                    status: 401,
                });
            }
            if (typeof user === "string" || !("id" in user)) {
                return new Response(JSON.stringify({ error: "Invalid user payload" }), {
                    status: 400,
                });
            }
            const { name } = await req.body;
            await prisma_1.prisma.user.update({
                where: { id: user.id },
                data: { name },
            });
            return new Response(JSON.stringify({ message: "User name updated" }), {
                status: 200,
            });
        }
        catch (error) {
            console.error(error);
            return res.status(500).json({ error: "Internal server error" });
        }
    },
    // ✅ get user balance
    balance: async (req, res) => {
        try {
            const userId = req.userId;
            if (!userId) {
                return res.status(401).json({ error: "Unauthorized", success: false });
            }
            const user = await prisma_1.prisma.user.findUnique({ where: { id: userId } });
            // get user balance
            // const availableBalance = await prisma.profile.findUnique({
            //   where: { userId: userId },
            //   select: { currentBalance: true },
            // });
            const userInfo = await prisma_1.prisma.user.findUnique({
                where: { id: userId },
                select: {
                    name: true,
                    phone: true,
                    profile: {
                        select: {
                            currentBalance: true,
                            vipName: true,
                            referralIncome: true,
                        },
                    },
                },
            });
            //get total withdraw amount
            const totalWithdraw = await prisma_1.prisma.transaction.aggregate({
                where: { userId: userId, type: "WITHDRAW" },
                _sum: { amount: true },
            });
            //get total deposit amount
            const totalDeposit = await prisma_1.prisma.transaction.aggregate({
                where: { userId: userId, type: "DEPOSIT" },
                _sum: { amount: true },
            });
            //get total commission amount
            const totalCommission = await prisma_1.prisma.transaction.aggregate({
                where: { userId: userId, type: "COMMISSION" },
                _sum: { amount: true },
            });
            // get user profile
            return res.json({
                userInfo,
                totalWithdraw,
                totalDeposit,
                totalCommission,
            });
        }
        catch (error) {
            if (error instanceof Error) {
                return res.status(500).json([{ error: error.message }]); // ✅ Use res.json()
            }
        }
    },
    //  ✅ deposit balance
    deposit: async (req, res) => {
        try {
            const userId = req.userId;
            if (!userId) {
                return res.status(401).json({ error: "Unauthorized", success: false });
            }
            // Validate inputs first (before any DB calls)
            const amount = Number(req.body.amount);
            const paymentMethod = req.body.paymentMethod;
            const depositType = req.body.depositType;
            if (isNaN(amount) || amount <= 0) {
                return res
                    .status(400)
                    .json({ error: "Invalid amount", success: false });
            }
            if (!paymentMethod || !depositType) {
                return res
                    .status(400)
                    .json({ error: "Missing required fields", success: false });
            }
            // Handle file upload
            let receiptFile = null;
            const files = req.files;
            if (Array.isArray(files) && files.length > 0) {
                receiptFile = `${files[0].filename}`;
            }
            else if (files?.receipt?.[0]) {
                receiptFile = `${files.receipt[0].filename}`;
            }
            if (!receiptFile) {
                return res
                    .status(400)
                    .json({ error: "Receipt file is required", success: false });
            }
            // Generate idempotency key
            const idempotencyKey = req.body.idempotencyKey || `deposit-${userId}-${Date.now()}`;
            // Use transaction for atomic operations
            const result = await prisma_1.prisma.$transaction(async (tx) => {
                // 🔥 CRITICAL: Check for existing order INSIDE the transaction
                const existingOrder = await tx.order.findFirst({
                    where: {
                        userId,
                        idempotencyKey,
                    },
                });
                if (existingOrder) {
                    // Return the existing order instead of throwing
                    return existingOrder;
                }
                // Get only needed settings and user data in parallel
                const [settings, userProfile, pendingOrdersCount, vipPrice] = await Promise.all([
                    tx.setting.findFirst({
                        select: { minDeposit: true },
                    }),
                    tx.profile.findUnique({
                        where: { userId },
                        select: { currentBalance: true, vipName: true },
                    }),
                    tx.order.count({
                        where: {
                            userId,
                            status: "PENDING",
                        },
                    }),
                    tx.vip.findMany({
                        orderBy: { price: "asc" },
                        select: { price: true, name: true },
                    }),
                ]);
                //  handle vips price in array
                const vipPrices = vipPrice?.map((vip) => vip.price) || [];
                const userBalance = userProfile?.currentBalance || 0;
                const minDeposit = settings?.minDeposit || 0;
                // current user vip price
                const userVip = vipPrice?.find((v) => v.name === userProfile?.vipName);
                const currentVipName = userProfile?.vipName || "";
                if (currentVipName === "0") {
                    if (amount < minDeposit) {
                        throw new Error(`Minimum deposit amount is ${minDeposit}`);
                    }
                }
                // Check for uncompleted orders
                if (pendingOrdersCount >= 1) {
                    throw new Error("You have a pending request");
                }
                const userVipPrice = userVip?.price || 0;
                // check userBalance + deposit amount should be one of vip price in the array
                if (!vipPrices.includes(userBalance + amount + Number(userVipPrice))) {
                    throw new Error(`Your current balance is ${userBalance}. You can't deposit ${amount} here. `);
                }
                // Create the order
                const order = await tx.order.create({
                    data: {
                        userId,
                        type: "DEPOSIT",
                        status: "PENDING",
                        depositType,
                        paymentMethod,
                        amount,
                        receipt: receiptFile,
                        idempotencyKey,
                    },
                    select: {
                        id: true,
                        amount: true,
                        status: true,
                        createdAt: true,
                    },
                });
                return order;
            }, {
                isolationLevel: client_1.Prisma.TransactionIsolationLevel.Serializable,
                timeout: 10000,
            });
            return res.json({
                message: result.id
                    ? "Order created successfully"
                    : "Order already processed",
                receipt: receiptFile,
                data: result,
                success: true,
            });
        }
        catch (error) {
            // Handle Prisma unique constraint errors (idempotency duplicates)
            if (error instanceof client_1.Prisma.PrismaClientKnownRequestError) {
                if (error.code === "P2002") {
                    // This would only happen if two requests somehow bypass the check
                    return res
                        .status(409)
                        .json({ error: "Duplicate request detected", success: false });
                }
            }
            console.error("Deposit error:", error);
            if (error instanceof Error) {
                return res.status(400).json({ error: error.message, success: false });
            }
            return res
                .status(500)
                .json({ error: "Internal server error", success: false });
        }
    },
    //  ✅ get deposit orders
    depositOrders: async (req, res) => {
        try {
            const userId = req.userId;
            if (!userId) {
                return res.status(401).json({ error: "Unauthorized", success: false });
            }
            // fetch user orders
            const orders = await prisma_1.prisma.order.findMany({
                where: { userId: userId, type: "DEPOSIT" },
                orderBy: { createdAt: "desc" },
                select: {
                    status: true,
                    paymentMethod: true,
                    amount: true,
                    createdAt: true,
                },
            });
            return res.json({ orders, success: true });
        }
        catch (error) {
            if (error instanceof Error) {
                return res.status(500).json([{ error: error.message }]); // ✅ Use res.json()
            }
        }
    },
    //  ✅ get referrals
    referrals: async (req, res) => {
        try {
            const userId = req.userId;
            if (!userId) {
                return res.status(401).json({ error: "Unauthorized", status: 401 });
            }
            const userProfile = await prisma_1.prisma.user.findUnique({
                where: { id: userId },
                select: {
                    id: true,
                    inviteCode: true,
                },
            });
            if (!userProfile) {
                return res.status(404).json({ error: "User not found", status: 404 });
            }
            // fetch number of referrals
            const referrals = await prisma_1.prisma.user.findMany({
                where: { invitedBy: userProfile.inviteCode },
            });
            //fetch number of active referrals
            const activeReferrals = await prisma_1.prisma.user.findMany({
                where: { invitedBy: userProfile.inviteCode, status: "1" },
            });
            // fetch latest 5 commission transactions
            const commissionTransactions = await prisma_1.prisma.transaction.findMany({
                where: { userId: userProfile.id, type: "COMMISSION" },
                orderBy: { createdAt: "desc" },
                take: 5,
            });
            return res.json({
                success: true,
                data: {
                    referrals: referrals.length,
                    activeReferrals: activeReferrals.length,
                    commissionTransactions,
                    invitationCode: userProfile.inviteCode,
                },
            });
        }
        catch (error) {
            return res.json({ error: `Internal server error ${error}`, status: 500 });
        }
    },
    //  ✅ withdraw
    withdraw: async (req, res) => {
        try {
            const userId = req.userId;
            if (!userId) {
                return res.status(401).json({ error: "Unauthorized", success: false });
            }
            // Validate inputs first
            const { amount, method, accountOwner, useAccount, phone } = req.body;
            const withdrawAmount = Number(amount);
            if (isNaN(withdrawAmount) || withdrawAmount <= 0) {
                return res
                    .status(400)
                    .json({ error: "Invalid amount", success: false });
            }
            if (!method || !accountOwner || !useAccount) {
                return res
                    .status(400)
                    .json({ error: "Missing required fields", success: false });
            }
            // Generate idempotency key (should come from client in production)
            const idempotencyKey = req.body.idempotencyKey || `withdraw-${userId}-${Date.now()}`;
            // Use transaction for atomic operations
            const result = await prisma_1.prisma.$transaction(async (tx) => {
                // 🔥 Check for existing withdrawal with same idempotency key
                const existingOrder = await tx.order.findFirst({
                    where: {
                        userId,
                        idempotencyKey,
                    },
                });
                if (existingOrder) {
                    return existingOrder;
                }
                // Lock the user profile to prevent concurrent modifications
                const userProfile = await tx.user.findUnique({
                    where: { id: userId },
                    select: {
                        id: true,
                        profile: {
                            select: {
                                currentBalance: true,
                            },
                        },
                    },
                });
                if (!userProfile?.profile) {
                    throw new Error("User profile not found");
                }
                // Get settings and pending withdrawals
                const [siteSettings, pendingWithdrawals] = await Promise.all([
                    tx.setting.findFirst({
                        select: {
                            minWithdraw: true,
                            maxWithdraw: true,
                        },
                    }),
                    // Use findMany with lock instead of count to prevent race conditions
                    tx.order.findMany({
                        where: {
                            userId,
                            status: "PENDING",
                            type: "WITHDRAW",
                        },
                    }),
                ]);
                const currentBalance = userProfile.profile.currentBalance;
                const minWithdraw = siteSettings?.minWithdraw || 0;
                const maxWithdraw = siteSettings?.maxWithdraw || 0;
                // Validate amount against limits
                if (withdrawAmount < minWithdraw) {
                    throw new Error(`Minimum withdrawal amount is ${minWithdraw}`);
                }
                if (withdrawAmount > maxWithdraw) {
                    throw new Error(`Maximum withdrawal amount is ${maxWithdraw}`);
                }
                // Check for pending withdrawals
                if (pendingWithdrawals.length >= 1) {
                    throw new Error("You have a pending withdrawal request");
                }
                // Check sufficient balance
                if (currentBalance < withdrawAmount) {
                    throw new Error("Insufficient balance");
                }
                // Create withdrawal order with idempotency key
                const order = await tx.order.create({
                    data: {
                        userId,
                        amount: withdrawAmount,
                        type: "WITHDRAW",
                        paymentMethod: method,
                        accountOwner,
                        useAccount,
                        phone,
                        status: "PENDING",
                        idempotencyKey, // Store the key
                    },
                    select: {
                        id: true,
                        amount: true,
                        status: true,
                        createdAt: true,
                        paymentMethod: true,
                    },
                });
                // Update balance
                await tx.profile.update({
                    where: { userId },
                    data: {
                        currentBalance: {
                            decrement: withdrawAmount,
                        },
                    },
                });
                return order;
            }, {
                isolationLevel: client_1.Prisma.TransactionIsolationLevel.Serializable,
                timeout: 10000,
            });
            return res.json({
                message: result.id
                    ? "Withdrawal request created successfully"
                    : "Request already processed",
                data: result,
                success: true,
            });
        }
        catch (error) {
            // Handle Prisma unique constraint errors
            if (error instanceof client_1.Prisma.PrismaClientKnownRequestError) {
                if (error.code === "P2002") {
                    return res
                        .status(409)
                        .json({
                        error: "Duplicate withdrawal request detected",
                        success: false,
                    });
                }
            }
            console.error("Withdraw error:", error);
            if (error instanceof Error) {
                return res.status(400).json({ error: error.message, success: false });
            }
            return res
                .status(500)
                .json({ error: "Internal server error", success: false });
        }
    },
    //  ✅ get withdraw orders
    withdrawOrders: async (req, res) => {
        try {
            const userId = req.userId;
            if (!userId) {
                return res.status(401).json({ error: "Unauthorized", success: false });
            }
            // fetch user orders
            const orders = await prisma_1.prisma.order.findMany({
                where: { userId: userId, type: "WITHDRAW" },
                orderBy: { createdAt: "desc" },
                select: {
                    status: true,
                    paymentMethod: true,
                    amount: true,
                    createdAt: true,
                },
            });
            return res.json({ orders, success: true });
        }
        catch (err) {
            console.error("Deposit Error:", err);
            return res.json({ error: "Server error", status: 500 });
        }
    },
    //  ✅ update VIP
    vip: async (req, res) => {
        try {
            const body = await req.body;
            const { vipLevel } = body;
            // const cookie = req.cookies.get("token");
            // if (!cookie) {
            //   return res.json({ error: "Unauthorized" }, { status: 401 });
            // }
            // const token = cookie.value;
            // get all settings
            const settings = await prisma_1.prisma.setting.findFirst();
            const commission = settings?.commission;
            // Validate request body
            if (!vipLevel || typeof vipLevel !== "string") {
                return res.json({ error: "VIP level is required", status: 400 });
            }
            const { token } = req.body;
            const user = await (0, jwt_1.verifyToken)(token);
            // Check authentication
            if (!user || !user.id) {
                return res.json({ error: "Unauthorized", status: 401 });
            }
            // const userData = await prisma.user.findUnique({ where: { id: user?.id } });
            // find user profile by user id
            const userProfile = await prisma_1.prisma.profile.findUnique({
                where: { userId: user.id },
            });
            // Check user profile exists
            if (!userProfile) {
                return res.json({ error: "User profile not found", status: 400 });
            }
            // Find VIP level
            const vip = await prisma_1.prisma.vip.findUnique({
                where: { name: vipLevel },
            });
            if (!vip) {
                return res.json({ error: "Invalid VIP level", status: 400 });
            }
            // Check balance
            if (userProfile.currentBalance < vip.price) {
                return res.json({ error: "Not enough balance", status: 400 });
            }
            // calculate commission
            // const commissionAmount = (Number(commission) / 100) * vip.price;
            // Update user VIP level
            const updatedUser = await prisma_1.prisma.user.update({
                where: { id: user.id },
                data: {
                    status: "1",
                    profile: {
                        update: {
                            vip: { connect: { name: vipLevel } },
                            currentBalance: {
                                decrement: vip.price,
                            },
                        },
                    },
                    transactions: {
                        create: {
                            type: "VIP",
                            amount: vip.price,
                        },
                    },
                },
                include: {
                    profile: {
                        include: {
                            vip: true,
                        },
                    },
                },
            });
            // update inviter balance and commission
            // if (userData?.invitedBy && userData?.status === "0") {
            //   await prisma.user.update({
            //     where: { inviteCode: userData?.invitedBy },
            //     data: {
            //       profile: {
            //         update: {
            //           currentBalance: {
            //             increment: commissionAmount,
            //           },
            //         },
            //       },
            //       transactions: {
            //         create: {
            //           type: "COMMISSION",
            //           amount: commissionAmount,
            //         },
            //       },
            //     },
            //   });
            // }
            return res.json({
                success: true,
                message: "VIP level updated successfully",
                data: updatedUser,
            });
        }
        catch (error) {
            console.error("VIP Update Error:", error);
            return res.json({ error: "Internal server error", status: 500 });
        }
    },
    // ✅ get task info
    taskInfo: async (req, res) => {
        try {
            const userId = req.userId;
            const { searchParams } = new URL(req.url, `${process.env.API_URL}`);
            const timeIsUp = searchParams.get("timeIsUp") === "true";
            if (!userId) {
                return res
                    .status(400)
                    .json({ error: "User ID is required", status: 400 });
            }
            // Use transaction to prevent race conditions
            const result = await prisma_1.prisma.$transaction(async (tx) => {
                // Get user with tasks and profile in one query
                const userData = await tx.user.findUnique({
                    where: { id: userId },
                    select: {
                        profile: {
                            select: {
                                vip: {
                                    select: {
                                        name: true,
                                        dailyIncome: true,
                                        incomePerTask: true,
                                        commission: true,
                                    },
                                },
                                taskStatus: true,
                            },
                        },
                        task: {
                            select: {
                                id: true,
                                taskInfoId: true,
                                status: true,
                                updatedAt: true,
                                userId: true,
                            },
                        },
                    },
                });
                if (!userData) {
                    throw new Error("User not found");
                }
                // If no tasks exist, create them
                if (userData.task.length === 0) {
                    const taskInfoList = await tx.taskInfo.findMany({
                        select: { id: true },
                    });
                    if (taskInfoList.length > 0) {
                        await tx.task.createMany({
                            data: taskInfoList.map((taskInfo) => ({
                                userId,
                                taskInfoId: taskInfo.id,
                                status: "0",
                            })),
                            skipDuplicates: true,
                        });
                        // Refresh tasks after creation
                        userData.task = await tx.task.findMany({
                            where: { userId },
                            select: {
                                id: true,
                                taskInfoId: true,
                                status: true,
                                updatedAt: true,
                                userId: true,
                            },
                        });
                    }
                }
                const latestTask = userData.task.reduce((prev, current) => {
                    return prev.updatedAt > current.updatedAt ? prev : current;
                });
                // check all tasks status
                const allTasksCompleted = userData.task.every((task) => task.status === "1");
                const now = new Date();
                const twentyFourHoursAgo = new Date(now.getTime() - 24 * 60 * 60 * 1000);
                // Handle task expiration - only update if needed
                if (timeIsUp && userData.task.length > 0) {
                    // is latest task updated more than 24 hours ago
                    // Find tasks that were updated more than 24 hours ago and aren't already "0"
                    const expiredTasks = userData.task.filter((task) => task.status !== "0" && task.updatedAt < twentyFourHoursAgo);
                    if (latestTask.updatedAt < twentyFourHoursAgo && allTasksCompleted) {
                        await tx.task.updateMany({
                            where: {
                                userId,
                                id: { in: expiredTasks.map((t) => t.id) },
                                status: { not: "0" },
                            },
                            data: { status: "0" },
                        });
                        // Update local task status
                        expiredTasks.forEach((task) => {
                            task.status = "0";
                        });
                    }
                }
                return userData;
            });
            return res.json({ success: true, data: result });
        }
        catch (error) {
            console.error("Get Profile Error:", error);
            return res.status(500).json({ error: "Server error", status: 500 });
        }
    },
    // ✅ make task
    makeTask: async (req, res) => {
        const { taskInfoId } = req.body;
        const id = Number(taskInfoId);
        const userId = req.userId;
        if (!userId) {
            return res.json({ error: "Unauthorized", status: 401 });
        }
        try {
            // 1. Get user with profile and VIP in single query
            const user = await prisma_1.prisma.user.findUnique({
                where: { id: userId },
                include: {
                    profile: {
                        include: {
                            vip: true,
                        },
                    },
                },
            });
            if (!user?.profile) {
                throw new Error("User profile not found");
            }
            const incomePerTask = user.profile.vip?.incomePerTask || 0;
            const currentBalance = user.profile.currentBalance;
            // 2. Check for existing task
            const existingTask = await prisma_1.prisma.task.findUnique({
                where: {
                    userId_taskInfoId: { userId, taskInfoId: id },
                },
            });
            if (existingTask) {
                if (existingTask.status === "0") {
                    const result = await processTaskCompletion(prisma_1.prisma, user, incomePerTask, currentBalance, id);
                    return res.json(result);
                }
                throw new Error("You've already completed this task today");
            }
            // For new task
            // const result = await processNewTask(
            //   prisma,
            //   user,
            //   incomePerTask,
            //   currentBalance,
            //   id
            // );
            // return res.json(result);
        }
        catch (error) {
            if (error instanceof Error) {
                return res.status(400).json({
                    error: error.message,
                    success: false,
                });
            }
            return res.status(500).json({
                error: "Failed to process task",
                success: false,
            });
        }
    },
};
// VIP upgrade helper function
const handleVipUpgrade = async (tx, user, newBalance) => {
    const allVips = await tx.vip.findMany({
        orderBy: { price: "asc" },
    });
    if (!allVips.length)
        return;
    // 🎯 Get current VIP info (MISSING IN YOUR CODE)
    const currentVip = allVips.find((v) => v.name === user.profile.vip?.name);
    const currentVipPrice = currentVip ? currentVip.price : 0;
    const currentVipLevel = Number(user.profile.vip?.name || "0");
    // 🎯 Check purchase history (MISSING IN YOUR CODE)
    const existingVipTransactions = await tx.transaction.findMany({
        where: {
            userId: user.id,
            type: "VIP",
        },
    });
    const hasPurchasedAnyVip = existingVipTransactions.length > 0;
    // 🎯 Use the same calculatedBalance logic as deposits
    const calculatedBalance = newBalance + currentVipPrice;
    // Find highest affordable VIP
    const affordableVips = allVips.filter((v) => v.price <= calculatedBalance);
    const highestAffordableVip = affordableVips.sort((a, b) => b.price - a.price)[0] || null;
    let finalBalance = newBalance;
    let vipTransactionCreated = false;
    let shouldUpgradeVip = false;
    // 🎯 SMART VIP LOGIC (same as deposits)
    if (highestAffordableVip) {
        const highestVipLevel = Number(highestAffordableVip.name);
        // Case 1: Upgrade to higher VIP
        if (highestVipLevel > currentVipLevel) {
            shouldUpgradeVip = true;
            finalBalance = calculatedBalance - highestAffordableVip.price;
            await tx.transaction.create({
                data: {
                    type: "VIP",
                    amount: highestAffordableVip.price,
                    userId: user.id,
                },
            });
            vipTransactionCreated = true;
        }
        // Case 2: Purchase current VIP for first time
        else if (highestVipLevel === currentVipLevel &&
            !hasPurchasedAnyVip &&
            highestAffordableVip.price > 0) {
            shouldUpgradeVip = true;
            finalBalance = calculatedBalance - highestAffordableVip.price;
            await tx.transaction.create({
                data: {
                    type: "VIP",
                    amount: highestAffordableVip.price,
                    userId: user.id,
                },
            });
            vipTransactionCreated = true;
        }
    }
    // Update profile
    const profileUpdateData = {
        currentBalance: finalBalance,
    };
    if (shouldUpgradeVip) {
        profileUpdateData.vipName = highestAffordableVip.name;
    }
    await tx.profile.update({
        where: { userId: user.id },
        data: profileUpdateData,
    });
    return {
        vipUpgraded: shouldUpgradeVip,
        vipLevel: shouldUpgradeVip
            ? highestAffordableVip.name
            : user.profile.vip?.name,
        vipPrice: vipTransactionCreated ? highestAffordableVip.price : 0,
        finalBalance,
        vipTransactionCreated,
    };
};
// Helper function for task completion
const processTaskCompletion = async (prisma, user, incomePerTask, currentBalance, taskInfoId) => {
    return await prisma.$transaction(async (tx) => {
        // Update existing task
        const result = await tx.task.update({
            where: { userId_taskInfoId: { userId: user.id, taskInfoId } },
            data: { status: "1" },
        });
        const newBalance = currentBalance + incomePerTask;
        // 🚨 REMOVED: Don't update balance here - let handleVipUpgrade do it
        // Create transaction record
        await tx.transaction.create({
            data: {
                userId: user.id,
                type: "TASK",
                amount: incomePerTask,
            },
        });
        // Handle VIP upgrade (this will update balance appropriately)
        const vipResult = await handleVipUpgrade(tx, user, newBalance);
        // Update total income separately (not affected by VIP purchase)
        await tx.profile.update({
            where: { userId: user.id },
            data: {
                totalIncome: { increment: incomePerTask },
            },
        });
        return {
            success: true,
            result,
            vipUpgrade: vipResult || null,
        };
    });
};
// Helper function for new task creation
// const processNewTask = async (
//   prisma: any,
//   user: any,
//   incomePerTask: number,
//   currentBalance: number,
//   taskInfoId: number
// ) => {
//   return await prisma.$transaction(async (tx: any) => {
//     // Create new task
//     const result = await tx.task.create({
//       data: {
//         userId: user.id,
//         taskInfoId: taskInfoId,
//         status: "1",
//       },
//     });
//     const newBalance = currentBalance + incomePerTask;
//     // 🚨 REMOVED: Don't update balance here - let handleVipUpgrade do it
//     // Create transaction record
//     await tx.transaction.create({
//       data: {
//         userId: user.id,
//         type: "TASK",
//         amount: incomePerTask,
//       },
//     });
//     // Handle VIP upgrade (this will update balance appropriately)
//     const vipResult = await handleVipUpgrade(tx, user, newBalance);
//     // Update total income separately (not affected by VIP purchase)
//     await tx.profile.update({
//       where: { userId: user.id },
//       data: {
//         totalIncome: { increment: incomePerTask },
//       },
//     });
//     return {
//       success: true,
//       result,
//       vipUpgrade: vipResult || null,
//     };
//   });
// };
