"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.systemController = void 0;
const prisma_1 = require("../lib/prisma");
exports.systemController = {
    // ✅ get site settings
    getSetting: async (req, res) => {
        try {
            const siteSettings = await prisma_1.prisma.setting.findFirst();
            if (!siteSettings) {
                return res.json({ error: "Site settings not found", status: 404 });
            }
            return res.json(siteSettings);
        }
        catch (error) {
            console.error(error);
            return res.status(500).json({ error: "Internal server error" });
        }
    },
    // ✅ get banks
    getBanks: async (req, res) => {
        const type = new URL(req.url, `${process.env.API_URL}`).searchParams.get("type");
        try {
            const userId = req.userId;
            let whereClause = {};
            let limit = {};
            let userBalance = null;
            switch (type) {
                case "DEPOSIT":
                    whereClause = { status: { in: ["DEPOSIT", "BOTH"] } }; // ✅ DEPOSIT or BOTH
                    limit = { maxDeposit: true, minDeposit: true, dallyDepositLimit: true };
                    break;
                case "WITHDRAW":
                    whereClause = { status: { in: ["WITHDRAW", "BOTH"] } }; // ✅ WITHDRAW or BOTH
                    limit = { maxWithdraw: true, minWithdraw: true, dallyWithdrawLimit: true, withdrawFee: true };
                    break;
                default:
                    whereClause = {}; // All
                    limit = { maxDeposit: true, minDeposit: true, dallyDepositLimit: true, maxWithdraw: true, minWithdraw: true, dallyWithdrawLimit: true, withdrawFee: true };
                    break;
            }
            const banks = await prisma_1.prisma.bank.findMany({
                where: whereClause,
            });
            // select max deposit and withdraw amount
            const siteSettings = await prisma_1.prisma.setting.findFirst({
                select: limit
            });
            if (userId) {
                // select only user balance
                userBalance = await prisma_1.prisma.profile.findUnique({
                    where: { userId: userId },
                    select: { currentBalance: true, vipName: true },
                });
            }
            return res.json({ banks, siteSettings, userBalance, success: true });
        }
        catch (err) {
            console.error("Bank Get Error:", err);
            return res.json({ error: "Server error", status: 500 });
        }
    },
    // ✅ post vip levels
    vipLevels: async (req, res) => {
        try {
            const body = await req.body;
            // ✅ Create VIP
            const vip = await prisma_1.prisma.vip.create({ data: body });
            return res.json({
                message: "VIP created successfully",
                vip,
                status: 201,
            });
        }
        catch (err) {
            console.error("VIP Create Error:", err);
            return res.json({ error: "Server error", status: 500 });
        }
    },
    // ✅ get vip levels
    getVip: async (req, res) => {
        try {
            const userId = req.userId;
            const vips = await prisma_1.prisma.vip.findMany({ orderBy: { price: "asc" } });
            let userVip = null;
            if (userId) {
                // select only user vip name
                userVip = await prisma_1.prisma.user.findUnique({
                    where: { id: userId },
                    select: {
                        profile: {
                            select: { vipName: true, currentBalance: true },
                        },
                    },
                });
            }
            return res.json({ vips, userVip, success: true });
        }
        catch (error) {
            console.error(error);
            return res.status(500).json({ error: "Internal server error" });
        }
    },
    // ✅ get task info
    getTaskInfo: async (req, res) => {
        try {
            const taskInfo = await prisma_1.prisma.taskInfo.findMany({
                orderBy: { taskId: "asc" },
            });
            return res.json({ success: true, data: taskInfo });
        }
        catch (error) {
            console.error(error);
            return res.status(500).json({ error: "Internal server error" });
        }
    },
};
