"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.vipController = exports.taskInfoController = exports.settingController = exports.bankController = void 0;
const prisma_1 = require("../lib/prisma");
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
// Bank CRUD Operations
exports.bankController = {
    // Get all banks
    getAllBanks: async (req, res) => {
        try {
            const banks = await prisma_1.prisma.bank.findMany({
                orderBy: { id: "asc" },
            });
            return res.json({ success: true, data: banks });
        }
        catch (error) {
            console.error("Error fetching banks:", error);
            return res.status(500).json({ error: "Server error" });
        }
    },
    // Create bank
    createBank: async (req, res) => {
        try {
            const { bankName, accNumber, owner, status } = req.body;
            let logo = null;
            const files = req.files;
            if (Array.isArray(files)) {
                // req.files is File[] (single field array)
                if (files.length > 0) {
                    logo = `${files[0].filename}`;
                }
            }
            else if (files && typeof files === "object") {
                // req.files is { [fieldname: string]: File[] }
                const logoFiles = files.logo;
                if (logoFiles && logoFiles.length > 0) {
                    logo = `${logoFiles[0].filename}`;
                }
            }
            const bank = await prisma_1.prisma.bank.create({
                data: { bankName, accNumber, owner, status, logo },
            });
            return res.json({
                success: true,
                data: { ...bank, logoUrl: logo ? `/uploads/system${logo}` : null },
            });
        }
        catch (error) {
            console.error("Error creating bank:", error);
            return res.status(500).json({ error: "Server error" });
        }
    },
    // Update bank
    updateBank: async (req, res) => {
        try {
            const { id } = req.params;
            const { bankName, accNumber, owner, status } = req.body;
            let logo = null;
            const files = req.files;
            if (Array.isArray(files)) {
                // req.files is File[] (single field array)
                if (files.length > 0) {
                    logo = `${files[0].filename}`;
                }
            }
            else if (files && typeof files === "object") {
                // req.files is { [fieldname: string]: File[] }
                const logoFiles = files.logo;
                if (logoFiles && logoFiles.length > 0) {
                    logo = `${logoFiles[0].filename}`;
                }
            }
            const bank = await prisma_1.prisma.bank.update({
                where: { id: parseInt(id) },
                data: { bankName, accNumber, owner, status },
            });
            return res.json({ success: true, data: bank });
        }
        catch (error) {
            console.error("Error updating bank:", error);
            return res.status(500).json({ error: "Server error" });
        }
    },
    // Delete bank
    deleteBank: async (req, res) => {
        try {
            const { id } = req.params;
            await prisma_1.prisma.bank.delete({
                where: { id: parseInt(id) },
            });
            return res.json({ success: true, message: "Bank deleted successfully" });
        }
        catch (error) {
            console.error("Error deleting bank:", error);
            return res.status(500).json({ error: "Server error" });
        }
    },
};
// Settings CRUD Operations
exports.settingController = {
    // Get settings
    getSettings: async (req, res) => {
        try {
            const settings = await prisma_1.prisma.setting.findFirst();
            return res.json({ success: true, data: settings });
        }
        catch (error) {
            console.error("Error fetching settings:", error);
            return res.status(500).json({ error: "Server error" });
        }
    },
    // Create or update settings
    updateSettings: async (req, res) => {
        try {
            const { commission, minWithdraw, maxWithdraw, withdrawFee, dallyDepositLimit, dallyWithdrawLimit, maxDeposit, minDeposit, dallyDepositLimitAmount, dallyWithdrawLimitAmount, } = req.body;
            // Check if settings exist
            const existingSettings = await prisma_1.prisma.setting.findFirst();
            let settings;
            if (existingSettings && existingSettings.id) {
                // Update existing
                settings = await prisma_1.prisma.setting.update({
                    where: { id: existingSettings.id },
                    data: {
                        commission: commission || existingSettings?.commission || "",
                        minWithdraw: minWithdraw || existingSettings?.minWithdraw || 0,
                        maxWithdraw: maxWithdraw || existingSettings?.maxWithdraw || 0,
                        withdrawFee: withdrawFee || existingSettings?.withdrawFee || "0",
                        dallyDepositLimit: dallyDepositLimit || existingSettings?.dallyDepositLimit || 0,
                        dallyWithdrawLimit: dallyWithdrawLimit || existingSettings?.dallyWithdrawLimit || 0,
                        maxDeposit: maxDeposit || existingSettings?.maxDeposit || 0,
                        minDeposit: minDeposit || existingSettings?.minDeposit || 0,
                        dallyDepositLimitAmount: dallyDepositLimitAmount ||
                            existingSettings?.dallyDepositLimitAmount ||
                            0,
                        dallyWithdrawLimitAmount: dallyWithdrawLimitAmount ||
                            existingSettings?.dallyWithdrawLimitAmount ||
                            0,
                    },
                });
            }
            else {
                // Create new
                settings = await prisma_1.prisma.setting.create({
                    data: {
                        commission: commission || existingSettings?.commission || "",
                        minWithdraw: minWithdraw || existingSettings?.minWithdraw || 0,
                        maxWithdraw: maxWithdraw || existingSettings?.maxWithdraw || 0,
                        withdrawFee: withdrawFee || existingSettings?.withdrawFee || "0",
                        dallyDepositLimit: dallyDepositLimit || existingSettings?.dallyDepositLimit || 0,
                        dallyWithdrawLimit: dallyWithdrawLimit || existingSettings?.dallyWithdrawLimit || 0,
                        maxDeposit: maxDeposit || existingSettings?.maxDeposit || 0,
                        minDeposit: minDeposit || existingSettings?.minDeposit || 0,
                        dallyDepositLimitAmount: dallyDepositLimitAmount ||
                            existingSettings?.dallyDepositLimitAmount ||
                            0,
                        dallyWithdrawLimitAmount: dallyWithdrawLimitAmount ||
                            existingSettings?.dallyWithdrawLimitAmount ||
                            0,
                    },
                });
            }
            return res.json({ success: true, data: settings });
        }
        catch (error) {
            console.error("Error updating settings:", error);
            return res.status(500).json({ error: "Server error" });
        }
    },
};
// TaskInfo CRUD Operations
// In controllers/settingController.ts - Update TaskInfo section
exports.taskInfoController = {
    // Get all task info
    getAllTaskInfo: async (req, res) => {
        try {
            const taskInfo = await prisma_1.prisma.taskInfo.findMany({
                orderBy: { id: "asc" },
            });
            // Add full image URLs
            const taskInfoWithUrls = taskInfo.map((task) => ({
                ...task,
                logoUrl: task.logoDir
                    ? `/uploads/system/${path_1.default.basename(task.logoDir)}`
                    : null,
            }));
            return res.json({ success: true, data: taskInfoWithUrls });
        }
        catch (error) {
            console.error("Error fetching task info:", error);
            return res.status(500).json({ error: "Server error" });
        }
    },
    // Create task info with file upload
    createTaskInfo: async (req, res) => {
        try {
            const { taskId, taskName, description, incomePerTask } = req.body;
            let logoDir = null;
            const files = req.files;
            if (Array.isArray(files)) {
                // req.files is File[] (single field array)
                if (files.length > 0) {
                    logoDir = `${files[0].filename}`;
                }
            }
            else if (files && typeof files === "object") {
                // req.files is { [fieldname: string]: File[] }
                const logoFiles = files.logo;
                if (logoFiles && logoFiles.length > 0) {
                    logoDir = `${logoFiles[0].filename}`;
                }
            }
            // Verify file actually exists
            // const fileExists = fs.existsSync(req.file.path);
            const taskInfo = await prisma_1.prisma.taskInfo.create({
                data: {
                    taskId,
                    taskName,
                    description,
                    incomePerTask: parseInt(incomePerTask),
                    logoDir,
                },
            });
            return res.json({
                success: true,
                data: {
                    ...taskInfo,
                    logoUrl: logoDir ? `/uploads/system/${logoDir}` : null,
                },
            });
        }
        catch (error) {
            console.error("❌ Error creating task info:", error);
            return res.status(500).json({ error: "Server error" });
        }
    },
    // Update task info with file upload
    updateTaskInfo: async (req, res) => {
        try {
            const { id } = req.params;
            const { taskId, taskName, description, incomePerTask } = req.body;
            // Get current task info to check for existing logo
            const currentTask = await prisma_1.prisma.taskInfo.findUnique({
                where: { id: parseInt(id) },
            });
            let logoDir = null;
            const files = req.files;
            if (Array.isArray(files)) {
                // req.files is File[] (single field array)
                if (files.length > 0) {
                    logoDir = `${files[0].filename}`;
                }
            }
            else if (files && typeof files === "object") {
                // req.files is { [fieldname: string]: File[] }
                const logoFiles = files.logo;
                if (logoFiles && logoFiles.length > 0) {
                    logoDir = `${logoFiles[0].filename}`;
                }
            }
            const taskInfo = await prisma_1.prisma.taskInfo.update({
                where: { id: parseInt(id) },
                data: {
                    taskId,
                    taskName,
                    description,
                    incomePerTask: parseInt(incomePerTask),
                    logoDir,
                },
            });
            return res.json({
                success: true,
                data: {
                    ...taskInfo,
                    logoUrl: logoDir ? `/uploads/system/${logoDir}` : null,
                },
            });
        }
        catch (error) {
            console.error("Error updating task info:", error);
            return res.status(500).json({ error: "Server error" });
        }
    },
    // Delete task info
    deleteTaskInfo: async (req, res) => {
        try {
            const id = req.body.id;
            if (!id) {
                return res.status(400).json({ error: "Task info ID is required" });
            }
            // Get task info to delete associated file
            const taskInfo = await prisma_1.prisma.taskInfo.findUnique({
                where: { id: parseInt(id) },
            });
            // Delete associated file if exists
            if (taskInfo?.logoDir) {
                const filePath = path_1.default.join(__dirname, "../uploads/system", taskInfo.logoDir);
                if (fs_1.default.existsSync(filePath)) {
                    fs_1.default.unlinkSync(filePath);
                }
            }
            await prisma_1.prisma.taskInfo.delete({
                where: { id: parseInt(id) },
            });
            return res.json({
                success: true,
                message: "Task info deleted successfully",
            });
        }
        catch (error) {
            console.error("Error deleting task info:", error);
            return res.status(500).json({ error: "Server error" });
        }
    },
};
// VIP CRUD Operations
// In controllers/settingController.ts - Update VIP section
exports.vipController = {
    // Get all VIP levels
    getAllVips: async (req, res) => {
        try {
            const vips = await prisma_1.prisma.vip.findMany({
                orderBy: { price: "asc" },
            });
            // Add full image URLs
            const vipsWithUrls = vips.map((vip) => ({
                ...vip,
                logoUrl: vip.logoDir
                    ? `/uploads/system/${path_1.default.basename(vip.logoDir)}`
                    : null,
            }));
            return res.json({ success: true, data: vipsWithUrls });
        }
        catch (error) {
            console.error("Error fetching VIP levels:", error);
            return res.status(500).json({ error: "Server error" });
        }
    },
    // Create VIP level with file upload
    createVip: async (req, res) => {
        try {
            const { name, price, dailyIncome, incomePerTask, commission, description, dailyTask, } = req.body;
            let logoDir = null;
            const files = req.files;
            if (Array.isArray(files)) {
                // req.files is File[] (single field array)
                if (files.length > 0) {
                    logoDir = `${files[0].filename}`;
                }
            }
            else if (files && typeof files === "object") {
                // req.files is { [fieldname: string]: File[] }
                const logoFiles = files.logo;
                if (logoFiles && logoFiles.length > 0) {
                    logoDir = `${logoFiles[0].filename}`;
                }
            }
            const vip = await prisma_1.prisma.vip.create({
                data: {
                    name,
                    price: parseInt(price),
                    dailyIncome: parseInt(dailyIncome),
                    incomePerTask: parseInt(incomePerTask),
                    commission: parseInt(commission),
                    description,
                    logoDir,
                    dailyTask: dailyTask ? parseInt(dailyTask) : null,
                },
            });
            return res.json({
                success: true,
                data: {
                    ...vip,
                    logoUrl: logoDir ? `/uploads/system/${logoDir}` : null,
                },
            });
        }
        catch (error) {
            console.error("Error creating VIP level:", error);
            return res.status(500).json({ error: "Server error" });
        }
    },
    // Update VIP level with file upload
    updateVip: async (req, res) => {
        try {
            const { id } = req.params;
            const { name, price, dailyIncome, incomePerTask, commission, description, dailyTask, } = req.body;
            // Get current VIP to check for existing logo
            const currentVip = await prisma_1.prisma.vip.findUnique({
                where: { id: parseInt(id) },
            });
            let logoDir = currentVip?.logoDir;
            // If new file uploaded, update logoDir
            if (req.file) {
                // Delete old file if exists
                if (currentVip?.logoDir) {
                    const oldFilePath = path_1.default.join(__dirname, "../public/uploads/system", currentVip.logoDir);
                    if (fs_1.default.existsSync(oldFilePath)) {
                        fs_1.default.unlinkSync(oldFilePath);
                    }
                }
                logoDir = req.file.filename;
            }
            const vip = await prisma_1.prisma.vip.update({
                where: { id: parseInt(id) },
                data: {
                    name,
                    price: parseInt(price),
                    dailyIncome: parseInt(dailyIncome),
                    incomePerTask: parseInt(incomePerTask),
                    commission: parseInt(commission),
                    description,
                    logoDir,
                    dailyTask: dailyTask ? parseInt(dailyTask) : null,
                },
            });
            return res.json({
                success: true,
                data: {
                    ...vip,
                    logoUrl: logoDir ? `/uploads/system/${logoDir}` : null,
                },
            });
        }
        catch (error) {
            console.error("Error updating VIP level:", error);
            return res.status(500).json({ error: "Server error" });
        }
    },
    // Delete VIP level
    deleteVip: async (req, res) => {
        try {
            const { id } = req.params;
            // Get VIP to delete associated file
            const vip = await prisma_1.prisma.vip.findUnique({
                where: { id: parseInt(id) },
            });
            // Delete associated file if exists
            if (vip?.logoDir) {
                const filePath = path_1.default.join(__dirname, "../public/uploads/system", vip.logoDir);
                if (fs_1.default.existsSync(filePath)) {
                    fs_1.default.unlinkSync(filePath);
                }
            }
            await prisma_1.prisma.vip.delete({
                where: { id: parseInt(id) },
            });
            return res.json({
                success: true,
                message: "VIP level deleted successfully",
            });
        }
        catch (error) {
            console.error("Error deleting VIP level:", error);
            return res.status(500).json({ error: "Server error" });
        }
    },
};
