"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.authController = void 0;
const bcrypt_1 = __importDefault(require("bcrypt"));
const prisma_1 = require("../lib/prisma");
const jwt_1 = require("../lib/jwt");
function generateInviteCode() {
    return Math.floor(100000 + Math.random() * 900000).toString();
}
exports.authController = {
    // ✅ user registration
    register: async (req, res) => {
        try {
            const { phone, password, inviteCode } = req.body;
            // Validate required fields
            if (!phone || !password) {
                return res.status(400).json({
                    error: "Phone and password are required",
                    success: false,
                });
            }
            // Use transaction for atomic operations
            const result = await prisma_1.prisma.$transaction(async (tx) => {
                // Check if phone exists
                const existingUser = await tx.user.findUnique({ where: { phone } });
                if (existingUser) {
                    throw new Error("Phone already registered");
                }
                // Handle inviter validation
                let inviter = null;
                if (inviteCode?.trim()) {
                    inviter = await tx.user.findUnique({
                        where: { inviteCode: inviteCode.trim() },
                    });
                    if (!inviter) {
                        throw new Error("Invalid invitation code");
                    }
                }
                // Generate unique invite code
                const newInviteCode = await generateUniqueInviteCode();
                // Get default VIP
                const defaultVip = await tx.vip.findFirst({
                    orderBy: { id: "asc" },
                });
                // Hash password
                const hashedPassword = await bcrypt_1.default.hash(password, 12);
                // Create user with profile
                const newUser = await tx.user.create({
                    data: {
                        phone,
                        password: hashedPassword,
                        inviteCode: newInviteCode,
                        invitedBy: inviter?.inviteCode || null,
                        invitedAt: inviter ? new Date() : null,
                        profile: {
                            create: {
                                currentBalance: 0,
                                vipName: defaultVip?.name || null,
                                referralIncome: 0,
                            },
                        },
                    },
                    include: { profile: true },
                });
                // Update inviter's referral income
                // if (inviter) {
                //   await tx.profile.update({
                //     where: { userId: inviter.id },
                //     data: { referralIncome: { increment: 1000 } },
                //   });
                // }
                return { newUser, inviter };
            });
            // Generate JWT
            const token = (0, jwt_1.signToken)({
                id: result.newUser.id,
                phone: result.newUser.phone,
                role: result.newUser.role,
            });
            return res.status(201).json({
                success: true,
                message: "User registered successfully",
                token,
                yourInviteCode: result.newUser.inviteCode,
                referralFrom: result.inviter?.phone || null,
            });
        }
        catch (error) {
            console.error("Registration error:", error);
            return res.status(500).json({
                error: error instanceof Error ? error.message : "Registration failed",
                success: false,
            });
        }
    },
    // Helper function for unique invite code generation
    // ✅ user login
    signIn: async (req, res) => {
        try {
            const body = req.body;
            const { phone, password } = body;
            // ✅ Find user
            const user = await prisma_1.prisma.user.findUnique({ where: { phone } });
            if (!user) {
                return res.status(400).json({
                    error: "Invalid phone number",
                    success: false,
                });
            }
            // ✅ Compare password
            const validPassword = await bcrypt_1.default.compare(password, user.password);
            if (!validPassword) {
                return res.status(400).json({
                    error: "Invalid password",
                    success: false,
                });
            }
            // ✅ Generate JWT
            const token = (0, jwt_1.signToken)({
                id: user.id,
                phone: user.phone,
                role: user.role,
                name: user.name,
            });
            return res.status(200).json({
                message: "Login successful",
                token,
                success: true,
                //  user: {
                //    id: user.id,
                //    phone: user.phone,
                //    role: user.role,
                //    name: user.name,
                //  },
            });
        }
        catch (err) {
            if (err instanceof Error) {
                return res.json({ error: err.message, status: 500 });
            }
        }
    },
    changePassword: async (req, res) => {
        try {
            const body = await req.body;
            const { currentPassword, newPassword } = body;
            if (!currentPassword || !newPassword) {
                return res.json({
                    error: "Missing currentPassword or newPassword",
                    status: 400,
                });
            }
            const userId = req.userId;
            if (!userId) {
                return res.json({ error: "Unauthorized", status: 401 });
            }
            // check the user
            const existingUser = await prisma_1.prisma.user.findUnique({
                where: { id: userId },
            });
            if (!existingUser) {
                return res.json({ error: "User not found", status: 404 });
            }
            if (!currentPassword || !newPassword) {
                return res.json({
                    error: "Missing currentPassword or newPassword",
                    status: 400,
                });
            }
            // Check current password
            const isCurrentPasswordValid = await bcrypt_1.default.compare(currentPassword, existingUser.password);
            if (!isCurrentPasswordValid) {
                return res.json({ error: "Invalid current password", status: 401 });
            }
            // Hash new password
            const hashedPassword = await bcrypt_1.default.hash(newPassword, 10);
            // Update user password in database
            await prisma_1.prisma.user.update({
                where: { id: userId },
                data: { password: hashedPassword },
            });
            return res.json({
                message: "Password updated successfully",
                success: true,
            });
        }
        catch (err) {
            if (err instanceof Error) {
                return res.json({ error: err.message, status: 500 });
            }
        }
    },
};
async function generateUniqueInviteCode(maxAttempts = 10) {
    for (let attempt = 0; attempt < maxAttempts; attempt++) {
        const code = generateInviteCode();
        const exists = await prisma_1.prisma.user.findUnique({
            where: { inviteCode: code },
        });
        if (!exists)
            return code;
    }
    throw new Error("Failed to generate unique invite code");
}
