"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.adminController = void 0;
const prisma_1 = require("../lib/prisma");
const DEFAULT_PAGE_SIZE = 10;
const DEFAULT_PAGE = 1;
// server url
exports.adminController = {
    // ✅ get all users
    getUsers: async (req, res) => {
        try {
            const role = req.role;
            if (!role || role === "USER") {
                return res.json({ error: "Unauthorized", status: 401 });
            }
            const { page = "1", limit = "10", search = "",
            // sortBy = "createdAt",
            // sortOrder = "desc",
             } = req.query;
            const pageNum = parseInt(page);
            const limitNum = parseInt(limit);
            const searchTerm = search.trim();
            const skip = (pageNum - 1) * limitNum;
            let whereClause = { role: "USER" };
            if (searchTerm !== "") {
                whereClause.OR = [
                    { name: { contains: searchTerm } },
                    { phone: { contains: searchTerm } },
                    { inviteCode: { contains: searchTerm } },
                ];
            }
            const [users, totalCount] = await Promise.all([
                prisma_1.prisma.user.findMany({
                    where: whereClause,
                    select: {
                        // Use select instead of include for better performance
                        id: true,
                        name: true,
                        phone: true,
                        role: true,
                        status: true,
                        inviteCode: true,
                        invitedBy: true,
                        invitedAt: true,
                        createdAt: true,
                        profile: {
                            select: {
                                vip: { select: { name: true } },
                                currentBalance: true,
                            },
                        },
                        // Remove transactions and orders from initial fetch
                        // We'll load these details on demand
                    },
                    // order by profile vip name asc
                    orderBy: { profile: { vip: { name: "desc" } } },
                    skip,
                    take: limitNum,
                }),
                prisma_1.prisma.user.count({ where: whereClause }),
            ]);
            // Calculate basic stats without heavy relations
            const usersWithBasicStats = await Promise.all(users.map(async (user) => {
                // Get deposit total only (lightweight query)
                const depositTotal = await prisma_1.prisma.transaction.aggregate({
                    where: {
                        userId: user.id,
                        type: "DEPOSIT",
                    },
                    _sum: { amount: true },
                });
                // Get withdraw total only
                const withdrawTotal = await prisma_1.prisma.transaction.aggregate({
                    where: {
                        userId: user.id,
                        type: "WITHDRAW",
                    },
                    _sum: { amount: true },
                });
                return {
                    ...user,
                    totalDeposits: depositTotal._sum.amount || 0,
                    totalWithdraws: withdrawTotal._sum.amount || 0,
                    totalOrders: 0, // We'll load this in details if needed
                };
            }));
            return res.json({
                users: usersWithBasicStats,
                pagination: {
                    page: pageNum,
                    limit: limitNum,
                    total: totalCount,
                    pages: Math.ceil(totalCount / limitNum),
                },
                success: true,
            });
        }
        catch (error) {
            console.error("Error fetching users:", error);
            return res.status(500).json({ error: "Server error" });
        }
    },
    // ✅ get user details
    getUserDetails: async (req, res) => {
        try {
            const role = req.role;
            if (!role || role === "USER") {
                return res.json({ error: "Unauthorized", status: 401 });
            }
            const { id } = req.params; // User ID from URL params
            const user = await prisma_1.prisma.user.findUnique({
                where: { id },
                include: {
                    profile: {
                        select: {
                            vip: { select: { name: true } },
                            currentBalance: true,
                        },
                    },
                    transactions: {
                        select: {
                            id: true,
                            type: true,
                            amount: true,
                            createdAt: true,
                        },
                        orderBy: { createdAt: "desc" },
                        take: 50, // Limit transactions to prevent huge responses
                    },
                    orders: {
                        where: { type: "DEPOSIT", status: "COMPLETED" },
                        select: {
                            id: true,
                            amount: true,
                            status: true,
                            paymentMethod: true,
                            createdAt: true,
                        },
                        orderBy: { createdAt: "desc" },
                        take: 20, // Limit orders
                    },
                },
            });
            if (!user) {
                return res.status(404).json({ error: "User not found" });
            }
            // Calculate totals
            const totalDeposits = user.transactions.reduce((total, transaction) => {
                return transaction.type === "DEPOSIT"
                    ? total + transaction.amount
                    : total;
            }, 0);
            const totalWithdraws = user.transactions.reduce((total, transaction) => {
                return transaction.type === "WITHDRAW"
                    ? total + transaction.amount
                    : total;
            }, 0);
            const totalOrders = user.orders.reduce((total, order) => {
                return total + order.amount;
            }, 0);
            const userWithStats = {
                ...user,
                totalDeposits,
                totalWithdraws,
                totalOrders,
            };
            return res.json({
                user: userWithStats,
                success: true,
            });
        }
        catch (error) {
            console.error("Error fetching user details:", error);
            return res.status(500).json({ error: "Server error" });
        }
    },
    //   ✅ get user info
    userInfo: async (req, res) => {
        try {
            const role = req.role;
            if (!role || role === "USER") {
                return res.json({ error: "Unauthorized", status: 401 });
            }
            // if (!user) {
            //   return res.json({ error: "Unauthorized" }, { status: 401 });
            // }
            // fetch all orders for admin
            const totalUsers = await prisma_1.prisma.user.count({ where: { role: "USER" } });
            const activeUsers = await prisma_1.prisma.user.count({
                where: { role: "USER", status: "1" },
            });
            const totalWithdraws = await prisma_1.prisma.order.count({
                where: { type: "WITHDRAW", status: "COMPLETED" },
            });
            const pendingWithdraws = await prisma_1.prisma.order.count({
                where: { type: "WITHDRAW", status: "PENDING" },
            });
            const totalWithdrawAmounts = await prisma_1.prisma.order.aggregate({
                where: { type: "WITHDRAW", status: "COMPLETED" },
                _sum: { amount: true },
            });
            const totalDeposits = await prisma_1.prisma.order.count({
                where: { type: "DEPOSIT", status: "COMPLETED" },
            });
            const pendingDeposits = await prisma_1.prisma.order.count({
                where: { type: "DEPOSIT", status: "PENDING" },
            });
            const totalDepositAmounts = await prisma_1.prisma.order.aggregate({
                where: { type: "DEPOSIT", status: "COMPLETED" },
                _sum: { amount: true },
            });
            const numberOfUsersWithEachVipName = await prisma_1.prisma.profile.groupBy({
                // where user role is user
                where: {
                    user: {
                        role: "USER",
                    },
                },
                by: ["vipName"],
                _count: { id: true },
            });
            return res.json({
                totalUsers,
                activeUsers,
                totalWithdraws,
                pendingWithdraws,
                totalWithdrawAmounts,
                totalDeposits,
                pendingDeposits,
                totalDepositAmounts,
                numberOfUsersWithEachVipName,
                success: true,
            });
        }
        catch (error) {
            console.error("Error fetching orders:", error);
            return res.json({ error: "Server error", status: 500 });
        }
    },
    //   ✅ get user orders
    userOrders: async (req, res) => {
        try {
            const role = req.role;
            if (!role || role === "USER") {
                return res.json({ error: "Unauthorized", status: 401 });
            }
            // Validate authentication
            // const user = await authenticate(req);
            // if (!user) {
            //   return res.json({ error: "Unauthorized", status: 401 });
            // }
            //  const { searchParams } = new URL(req.url, `${process.env.API_URL}`);
            // Validate and parse pagination parameters
            // const page = Math.max(
            //   DEFAULT_PAGE,
            //   Number(
            //     new URL(req.url, `${process.env.API_URL}`).searchParams.get("page") ||
            //       DEFAULT_PAGE
            //   )
            // );
            // const skip = (page - 1) * DEFAULT_PAGE_SIZE;
            const { searchParams } = new URL(req.url, `${process.env.API_URL}`);
            const pageSize = Number(searchParams.get("pageSize") || DEFAULT_PAGE_SIZE);
            const page = Math.max(DEFAULT_PAGE, Number(searchParams.get("page") || DEFAULT_PAGE));
            const skip = (page - 1) * pageSize;
            let whereClause = {};
            const search = searchParams.get("search") || "";
            switch (search) {
                case "DEPOSIT":
                    whereClause = { depositType: "DEPOSIT", status: "PENDING" }; // ✅ DEPOSIT or BOTH
                    break;
                case "WITHDRAW":
                    whereClause = { type: "WITHDRAW", status: "PENDING" }; // ✅ WITHDRAW or BOTH
                    break;
                case "VIP_UPGRADE":
                    whereClause = { depositType: "VIP_UPGRADE", status: "PENDING" }; // ✅ WITHDRAW or BOTH
                    break;
                case " ":
                    whereClause = { depositType: "DEPOSIT", status: "PENDING" }; // ✅ WITHDRAW or BOTH
                    break;
                default:
                    whereClause = { depositType: "WITHDRAW", status: "PENDING" };
            }
            // Fetch orders with pagination
            const [ordersData, totalOrders] = await Promise.all([
                prisma_1.prisma.order.findMany({
                    skip,
                    take: pageSize,
                    orderBy: { createdAt: "desc" },
                    where: whereClause,
                    include: {
                        user: {
                            select: {
                                name: true,
                                phone: true,
                            },
                        },
                    },
                }),
                prisma_1.prisma.order.count(),
            ]);
            // Calculate total pages
            const totalPages = Math.ceil(totalOrders / DEFAULT_PAGE_SIZE);
            // In your GET function:
            const orders = ordersData.map((item) => {
                const { user, ...rest } = item;
                return {
                    ...rest,
                    id: rest.id.toString(),
                    name: user?.name || null,
                    phone: user?.phone || null,
                };
            });
            // Return response with pagination info
            return res.json({
                orders,
                totalPages,
                currentPage: page,
            });
        }
        catch (error) {
            console.error("Error fetching orders:", error);
            return res.json({ error: "Internal server error", status: 500 });
        }
    },
    //   ✅ approve order
    approveOrder: async (req, res) => {
        try {
            const role = req.role;
            if (!role || role === "USER") {
                return res.json({ error: "Unauthorized", status: 401 });
            }
            const { orderId, type, status } = req.body;
            if (!orderId || !type || !status) {
                return res.status(400).json({
                    error: "Missing required fields",
                    success: false,
                });
            }
            // 1. Get order with user data in single query
            const orderWithUser = await prisma_1.prisma.order.findUnique({
                where: { id: Number(orderId) },
                include: {
                    user: {
                        include: {
                            profile: true,
                        },
                    },
                },
            });
            if (!orderWithUser) {
                return res.status(400).json({
                    error: "Order not found",
                    success: false,
                });
            }
            if (!orderWithUser.user?.profile) {
                return res.status(400).json({
                    error: "User profile not found",
                    success: false,
                });
            }
            const { user } = orderWithUser;
            const userProfile = user.profile;
            // 2. Use transaction for data consistency
            const result = await prisma_1.prisma.$transaction(async (tx) => {
                // Update order status
                const updatedOrder = await tx.order.update({
                    where: { id: Number(orderId) },
                    data: { status },
                });
                // Handle completed deposits
                if (status === "COMPLETED" &&
                    type === "DEPOSIT" &&
                    (role === "SUPER_ADMIN" || role === "ADMIN")) {
                    await handleDepositCompletion(tx, user, userProfile, orderWithUser);
                }
                // Handle completed withdrawals
                if (status === "COMPLETED" && type === "WITHDRAW") {
                    await tx.transaction.create({
                        data: {
                            type: "WITHDRAW",
                            amount: Number(orderWithUser.amount),
                            userId: user.id,
                        },
                    });
                }
                // Handle rejected withdrawals
                if (status === "REJECTED" && type === "WITHDRAW") {
                    await tx.profile.update({
                        where: { userId: user.id },
                        data: {
                            currentBalance: { increment: Number(orderWithUser.amount) },
                        },
                    });
                }
                return updatedOrder;
            });
            return res.status(200).json({
                updatedOrder: result,
                success: true,
            });
        }
        catch (error) {
            console.error("Error approving order:", error);
            return res.status(500).json({
                error: "Internal server error",
                success: false,
                details: process.env.NODE_ENV === "development" ? error.message : undefined,
            });
        }
    },
    // ✅ delete user function
    deleteUser: async (req, res) => {
        try {
            const role = req.role;
            if (!role || role === "USER") {
                return res.json({ error: "Unauthorized", status: 401 });
            }
            const { id } = req.params;
            // First, check if user exists
            const user = await prisma_1.prisma.user.findUnique({
                where: { id },
                include: {
                    profile: true,
                    orders: true,
                    transactions: true,
                    task: true,
                },
            });
            if (!user) {
                return res
                    .status(404)
                    .json({ error: "User not found", success: false });
            }
            // Use transaction to ensure all related records are deleted
            await prisma_1.prisma.$transaction(async (tx) => {
                // Delete related records in the correct order
                // 1. Delete user's tasks
                if (user.task.length > 0) {
                    await tx.task.deleteMany({
                        where: { userId: id },
                    });
                }
                // 2. Delete user's profile
                if (user.profile) {
                    await tx.profile.delete({
                        where: { userId: id },
                    });
                }
                // 3. Delete user's transactions
                if (user.transactions.length > 0) {
                    await tx.transaction.deleteMany({
                        where: { userId: id },
                    });
                }
                // 4. Delete user's orders
                if (user.orders.length > 0) {
                    await tx.order.deleteMany({
                        where: { userId: id },
                    });
                }
                // 5. Finally delete the user
                await tx.user.delete({
                    where: { id },
                });
            });
            return res.json({
                message: "User and all related data deleted successfully",
                success: true,
            });
        }
        catch (error) {
            console.error("Error deleting user:", error);
            // Handle specific error cases
            if (error.code === "P2025") {
                return res.status(404).json({
                    error: "User not found",
                    success: false,
                });
            }
            return res.status(500).json({
                error: "Internal server error",
                success: false,
                details: process.env.NODE_ENV === "development" ? error.message : undefined,
            });
        }
    },
};
// Helper function outside the class
const handleDepositCompletion = async (tx, user, userProfile, order) => {
    try {
        const allVips = await tx.vip.findMany({ orderBy: { price: "asc" } });
        if (!allVips.length) {
            throw new Error("No VIPs found");
        }
        // Find current VIP info
        const currentVip = allVips.find((v) => v.name === userProfile.vipName);
        const currentVipPrice = currentVip ? currentVip.price : 0;
        const currentVipLevel = Number(userProfile.vipName || "0");
        const currentBalance = Number(userProfile.currentBalance);
        const depositAmount = Number(order.amount);
        const newBalance = currentBalance + depositAmount;
        const calculatedBalance = newBalance + currentVipPrice;
        // Check if user already purchased any VIP
        const existingVipTransactions = await tx.transaction.findMany({
            where: {
                userId: user.id,
                type: "VIP",
            },
        });
        const hasPurchasedAnyVip = existingVipTransactions.length > 0;
        // Find highest affordable VIP
        const affordableVips = allVips.filter((v) => v.price <= calculatedBalance);
        const highestAffordableVip = affordableVips.sort((a, b) => b.price - a.price)[0] || null;
        let finalBalance = newBalance;
        let vipTransactionCreated = false;
        let shouldUpgradeVip = false;
        // Smart VIP logic
        if (highestAffordableVip) {
            const highestVipLevel = Number(highestAffordableVip.name);
            // Case 1: Upgrade to higher VIP
            if (highestVipLevel > currentVipLevel) {
                shouldUpgradeVip = true;
                finalBalance = calculatedBalance - highestAffordableVip.price;
                await tx.transaction.create({
                    data: {
                        type: "VIP",
                        amount: highestAffordableVip.price,
                        userId: user.id,
                    },
                });
                vipTransactionCreated = true;
            }
            // Case 2: Purchase current VIP for first time
            else if (highestVipLevel === currentVipLevel &&
                !hasPurchasedAnyVip &&
                highestAffordableVip.price > 0) {
                shouldUpgradeVip = true;
                finalBalance = calculatedBalance - highestAffordableVip.price;
                await tx.transaction.create({
                    data: {
                        type: "VIP",
                        amount: highestAffordableVip.price,
                        userId: user.id,
                    },
                });
                vipTransactionCreated = true;
            }
        }
        // 
        if (highestAffordableVip) {
        }
        // Update profile
        const profileUpdateData = {
            currentBalance: finalBalance,
            totalDeposit: { increment: depositAmount },
        };
        if (shouldUpgradeVip) {
            profileUpdateData.vipName = highestAffordableVip.name;
        }
        await tx.profile.update({
            where: { userId: user.id },
            data: profileUpdateData,
        });
        // Activate user if first deposit
        if (user.status === "0") {
            await tx.user.update({
                where: { id: user.id },
                data: { status: "1" },
            });
        }
        // activate task
        // Create deposit transaction
        await tx.transaction.create({
            data: {
                type: "DEPOSIT",
                amount: depositAmount,
                userId: user.id,
            },
        });
        // Handle commission (first deposit only)
        if (user.invitedBy && user.status === "0") {
            const settings = await tx.setting.findFirst();
            const commission = settings?.commission || 0;
            const commissionAmount = (Number(commission) / 100) * depositAmount;
            const referrer = await tx.user.findUnique({
                where: { inviteCode: user.invitedBy },
                include: { profile: true },
            });
            if (referrer?.profile) {
                await tx.profile.update({
                    where: { userId: referrer.id },
                    data: {
                        currentBalance: { increment: commissionAmount },
                    },
                });
                await tx.transaction.create({
                    data: {
                        type: "COMMISSION",
                        amount: commissionAmount,
                        userId: referrer.id,
                    },
                });
            }
        }
        return {
            vipUpgraded: shouldUpgradeVip,
            vipLevel: shouldUpgradeVip
                ? highestAffordableVip.name
                : userProfile.vipName,
            vipPrice: vipTransactionCreated ? highestAffordableVip.price : 0,
            finalBalance,
            vipTransactionCreated,
        };
    }
    catch (error) {
        console.error("Error in handleDepositCompletion:", error);
        throw error;
    }
};
